import networkx as nx
import matplotlib.pyplot as plt
from torch_geometric.datasets import TUDataset

# === 1. 加载 TU Dataset ===
dataset = TUDataset(root='data', name='ENZYMES', use_node_attr=True)

# 选一张小图，随便拿第一张
data = dataset[5]

# === 2. 转 networkx ===
G = nx.Graph()
edge_list = data.edge_index.t().tolist()
G.add_edges_from([tuple(edge) for edge in edge_list])

# === 3. 找环基 ===
rings = nx.cycle_basis(G)
print(f"Found {len(rings)} rings: ", rings)

# === 4. 画图，环边高亮 ===
pos = nx.spring_layout(G, seed=42)  # 用 spring layout 比较美观

# 找环内所有的边集合
ring_edges = set()
for cycle in rings:
    n = len(cycle)
    for i in range(n):
        u = cycle[i]
        v = cycle[(i+1)%n]
        ring_edges.add(tuple(sorted((u,v))))

# 按是否在环中分颜色
edge_colors = []
for u, v in G.edges():
    if tuple(sorted((u,v))) in ring_edges:
        edge_colors.append('red')
    else:
        edge_colors.append('gray')

# 画节点
nx.draw_networkx_nodes(G, pos, node_size=300, node_color='lightblue', edgecolors='black')
# 画边
nx.draw_networkx_edges(G, pos, edge_color=edge_colors, width=2)
# 画标签
nx.draw_networkx_labels(G, pos)

plt.title("Example TU Graph with Ring Highlighted (MCB)")
plt.axis('off')
plt.tight_layout()
plt.tight_layout()
plt.savefig("tu_graph_with_rings5.png", dpi=300)

